/**  @file bta_laplace.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_laplace.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <bta_opencv_helper.h>
#include <bta_helper.h>

#ifndef BTA_EXCLUDE_FILTERS

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>


cv::Mat toPresentableMat(BTA_Channel *channel);

BTA_Status BFLTlaplaceInit(BTA_FltLaplaceConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    BTA_FltLaplaceInst *inst = (BTA_FltLaplaceInst *)calloc(1, sizeof(BTA_FltLaplaceInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->channelToProcess = config->channelToProcess;
    inst->channelIdResult = config->channelIdResult;
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTlaplaceClose(BTA_FltHandle *handle) {
    BTA_FltLaplaceInst **inst = (BTA_FltLaplaceInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTlaplaceApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltLaplaceInst *inst = (BTA_FltLaplaceInst *)handle;
    int chInd, x, y;
    int chCount = (*frame)->channelsLen;
    for (chInd = 0; chInd < chCount; chInd++) {
        BTA_Channel *channel = ((*frame)->channels)[chInd];
        if (channel->id == inst->channelToProcess) {
            uint8_t freeChannel = 0;
            if (channel->dataFormat == BTA_DataFormatRgb24 || channel->dataFormat == BTA_DataFormatRgb565) {
                BTA_Channel *channelTemp;
                BTA_Status status = BTAcolorToBw(channel, &channelTemp);
                if (status != BTA_StatusOk) {
                    return status;
                }
                channel = channelTemp;
                freeChannel = 1;
            }
            cv::Mat mat = BTAtoMat(channel);
            cv::Mat matFloat;
            mat.convertTo(matFloat, CV_32F);
            double min, max;
            cv::minMaxLoc(matFloat, &min, &max);
            if (min < max) {
                for (y = 0; y < matFloat.size().height; y++) {
                    for (x = 0; x < matFloat.size().width; x++) {
                        matFloat.at<float>(y, x) = (float)((matFloat.at<float>(y, x) - min) * 1024.0 / (max - min));
                    }
                }
            }
            cv::Mat matDerived;
            cv::Laplacian(matFloat, matDerived, CV_32F);
            if (inst->channelIdResult) {
                BTAinsertChannelIntoFrame(*frame, BTAtoChannel(inst->channelIdResult, BTA_UnitUnitLess, channel->integrationTime, channel->modulationFrequency, matDerived));
            }
            else {
                BTAinsertChannelIntoFrame(*frame, BTAtoChannel(channel->id, BTA_UnitUnitLess, channel->integrationTime, channel->modulationFrequency, matDerived));
                BTAremoveChannelFromFrame(*frame, channel);
            }
            if (freeChannel) {
                // we created a new intermediate channel before -> free it!
                BTAfreeChannel(&channel);
            }
        }
    }
    return BTA_StatusOk;
}
#endif